#ifndef __CHRONO_H__
#define __CHRONO_H__

#include <ctime>

/**
 * \namespace	Chrono 
 *
 * \brief	A small set of utilities encapsulating timing functions 
**/
namespace Chrono 
{

/**
* \class	Stopwatch
*
* \brief	A stopwatch
*
* \details	Allows to create a stopwatch, start, stop and getting time in various formats.
**/
class Stopwatch
{
public:
 /**
 * \fn	Stopwatch::Stopwatch();
 *
 * \brief	Default constructor. 
 *
**/
inline
Stopwatch();

 /**
 * \fn	void Stopwatch::Start();
 *
 * \brief	Starts the time counter. 
 *
**/
inline
void Start();

  /**
 * \fn	void Stopwatch::Stop();
 *
 * \brief	Stops the time counter. 
 *
**/
inline
void Stop();

  /**
 * \fn	void Stopwatch::Reset();
 *
 * \brief	Resets the time counter to 0 sec.
 *
**/
inline
void Reset();

  /**
 * \fn	double  Stopwatch::GetTimeInMilliseconds();
 *
 * \brief	Gets the time in milliseconds. 
 *
 * \return	The time in milliseconds. 
 *
**/
inline
double  GetTimeInMilliseconds();

		/**
		 * \fn	double Stopwatch::GetTimeInSeconds();
		 *
		 * \brief	Gets the time in seconds. 
		 *
		 * \return	The time in seconds. 
		 *
		**/
		inline
		double  GetTimeInSeconds();

		/**
		 * \fn	double Stopwatch::GetTimeInMinutes();
		 *
		 * \brief	Gets the time in minutes. 
		 *
		 * \return	The time in minutes. 
		 *
		**/
		inline
		double  GetTimeInMinutes();

		/**
		 * \fn	double Stopwatch::GetTimeInHours();
		 *
		 * \brief	Gets the time in hours. 
		 *
		 * \return	The time in hours. 
		 *
		**/
		inline
		double  GetTimeInHours();

		/**
		 * \fn	void Stopwatch::GetTime(int &aHours, int &aMinutes, int &aSeconds, int &aMilliseconds);
		 *
		 * \brief	Gets a time in a formated way
		 *
		 * \param [in,out]	aHours			a in hours. 
		 * \param [in,out]	aMinutes		a in minutes. 
		 * \param [in,out]	aSeconds		a in seconds. 
		 * \param [in,out]	aMilliseconds	a in milliseconds. 
		 *
		**/
		inline
		void    GetTime(int &aHours, int &aMinutes, int &aSeconds, int &aMilliseconds);

	private :
		unsigned int mStart;	///< The start time 
		unsigned int mAccumulator;  ///< The time accumulator
		bool mStopped;  ///< true if time counter is stopped

	};
}

inline
Chrono::Stopwatch::Stopwatch() : mAccumulator(0), mStopped(true)
{
}

inline
void Chrono::Stopwatch::Start()
{
	if(!mStopped)
		return;

	mStart = clock() - mAccumulator;
	mStopped = false;
}

inline
void Chrono::Stopwatch::Stop()
{
	if(mStopped)
		return;

	mAccumulator = clock() - mStart;
	mStopped = true;
}

inline
void Chrono::Stopwatch::Reset()
{
	Stop();
	mAccumulator = 0;
}

inline
double Chrono::Stopwatch::GetTimeInMilliseconds()
{
	return GetTimeInSeconds() * 1000.0;
}

inline
double Chrono::Stopwatch::GetTimeInSeconds()
{
	unsigned int lTime;
	if(mStopped)
		lTime = mAccumulator;
	else
		lTime = clock() - mStart;

	return lTime / (double)CLOCKS_PER_SEC;
}

inline
double Chrono::Stopwatch::GetTimeInMinutes()
{
	return GetTimeInSeconds() / 60.0;
}

inline
double Chrono::Stopwatch::GetTimeInHours()
{
	return GetTimeInSeconds() / 3600.0;
}

inline
void Chrono::Stopwatch::GetTime(int &aHours, int &aMinutes, int &aSeconds, int &aMilliseconds)
{
	double lRemainder = GetTimeInSeconds();

	aHours = (int)(lRemainder / 3600.0);
	lRemainder -= aHours;

	aMinutes = (int)(lRemainder / 60.0);
	lRemainder -= aMinutes;

	aSeconds = (int)lRemainder;
	lRemainder -= aSeconds;

	aMilliseconds = (int)lRemainder;

}

#endif // __CHRONO_H__